import React from "react";
import { useLanguage } from "@/contexts/LanguageContext";

type TaskAnalysisData = {
  taskStats: {
    totalTasks: number;
    completedTasks: number;
    pendingTasks: number;
    overdueTasks: number;
    recurringTasks: number;
    nonRecurringTasks: number;
    periodCompletionRate: number;
    totalPeriods: number;
    completedPeriods: number;
    tasksCreatedToday: number;
    tasksCreatedThisWeek: number;
    tasksCreatedThisMonth: number;
  };
  patternStats: Array<{
    pattern: string;
    total: number;
    completed: number;
    completionRate: number;
  }>;
  departmentStats: Array<{
    department: string;
    total: number;
    completed: number;
    overdue: number;
    nonConformities: number;
    completionRate: number;
  }>;
  technicianStats: Array<{
    technicianId: string;
    technicianName: string;
    technicianEmail: string;
    isExternal: boolean;
    companyName: string | null;
    total: number;
    completed: number;
    overdue: number;
    avgCompletionTime: number;
    completionRate: number;
  }>;
  nonConformityStats: {
    totalNonConformities: number;
    tasksWithNonConformities: number;
    nonConformityRate: number;
  };
};

interface TaskPerformanceTabProps {
  taskData: TaskAnalysisData;
}

export default function TaskPerformanceTab({
  taskData,
}: TaskPerformanceTabProps) {
  const { t } = useLanguage();
  const { technicianStats, patternStats } = taskData;

  // Sort technicians by average completion time (fastest first)
  const sortedBySpeed = [...technicianStats]
    .filter((t) => t.avgCompletionTime > 0)
    .sort((a, b) => a.avgCompletionTime - b.avgCompletionTime);

  // Sort technicians by completion rate
  const sortedByRate = [...technicianStats].sort(
    (a, b) => b.completionRate - a.completionRate
  );

  // Calculate overall averages
  const avgCompletionTime =
    sortedBySpeed.length > 0
      ? sortedBySpeed.reduce((sum, t) => sum + t.avgCompletionTime, 0) /
        sortedBySpeed.length
      : 0;

  const avgCompletionRate =
    technicianStats.length > 0
      ? technicianStats.reduce((sum, t) => sum + t.completionRate, 0) /
        technicianStats.length
      : 0;

  return (
    <>
      {/* Performance Overview */}
      <div className='grid grid-cols-1 md:grid-cols-3 gap-6 mb-8'>
        <div className='bg-white p-6 rounded-lg shadow'>
          <h3 className='text-sm font-medium text-gray-500 mb-2'>
            {t("analysis.taskAnalysis.avgCompletionTime") || "Average Completion Time"}
          </h3>
          <p className='text-3xl font-bold text-blue-600'>
            {avgCompletionTime > 0 ? avgCompletionTime.toFixed(1) : "—"}
          </p>
          <p className='text-xs text-gray-500 mt-1'>{t("analysis.taskAnalysis.days") || "days"}</p>
        </div>
        <div className='bg-white p-6 rounded-lg shadow'>
          <h3 className='text-sm font-medium text-gray-500 mb-2'>
            {t("analysis.taskAnalysis.avgCompletionRate") || "Average Completion Rate"}
          </h3>
          <p className='text-3xl font-bold text-green-600'>
            {avgCompletionRate.toFixed(1)}%
          </p>
          <p className='text-xs text-gray-500 mt-1'>{t("analysis.taskAnalysis.acrossAllTechnicians") || "across all technicians"}</p>
        </div>
        <div className='bg-white p-6 rounded-lg shadow'>
          <h3 className='text-sm font-medium text-gray-500 mb-2'>
            {t("analysis.taskAnalysis.activeTechnicians") || "Active Technicians"}
          </h3>
          <p className='text-3xl font-bold text-gray-900'>
            {technicianStats.length}
          </p>
          <p className='text-xs text-gray-500 mt-1'>{t("analysis.taskAnalysis.withAssignedTasks") || "with assigned tasks"}</p>
        </div>
      </div>

      {/* Fastest Technicians */}
      <div className='bg-white p-6 rounded-lg shadow mb-8'>
        <h3 className='text-lg font-semibold mb-4'>
          {t("analysis.taskAnalysis.fastestCompletionTimes") || "Fastest Completion Times"}
        </h3>
        {sortedBySpeed.length > 0 ? (
          <div className='overflow-x-auto'>
            <table className='min-w-full divide-y divide-gray-200'>
              <thead className='bg-gray-50'>
                <tr>
                  <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                    {t("analysis.taskAnalysis.technician") || "Technician"}
                  </th>
                  <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                    {t("analysis.taskAnalysis.type") || "Type"}
                  </th>
                  <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                    {t("analysis.taskAnalysis.avgTimeDays") || "Avg Time (Days)"}
                  </th>
                  <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                    {t("analysis.taskAnalysis.completed") || "Completed"}
                  </th>
                  <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                    {t("analysis.taskAnalysis.completionRateLabel") || "Completion Rate"}
                  </th>
                </tr>
              </thead>
              <tbody className='bg-white divide-y divide-gray-200'>
                {sortedBySpeed.slice(0, 10).map((tech) => (
                  <tr key={tech.technicianId}>
                    <td className='px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900'>
                      <div>
                        {tech.technicianName}
                        {tech.isExternal && (
                          <span className='ml-2 text-xs text-blue-600'>
                            [{t("analysis.taskAnalysis.external") || "External"}]
                          </span>
                        )}
                      </div>
                    </td>
                    <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                      {tech.isExternal ? (
                        <span className='text-blue-600'>
                          {tech.companyName || t("analysis.taskAnalysis.external") || "External"}
                        </span>
                      ) : (
                        <span className='text-gray-600'>{t("analysis.taskAnalysis.internal") || "Internal"}</span>
                      )}
                    </td>
                    <td className='px-6 py-4 whitespace-nowrap text-sm font-bold text-green-600'>
                      {tech.avgCompletionTime.toFixed(1)} {t("analysis.taskAnalysis.days") || "days"}
                    </td>
                    <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                      {tech.completed}
                    </td>
                    <td className='px-6 py-4 whitespace-nowrap text-sm'>
                      <span
                        className={`font-medium ${
                          tech.completionRate >= 80
                            ? "text-green-600"
                            : tech.completionRate >= 50
                            ? "text-orange-600"
                            : "text-red-600"
                        }`}
                      >
                        {tech.completionRate.toFixed(1)}%
                      </span>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        ) : (
          <p className='text-gray-500 text-center py-8'>
            {t("analysis.taskAnalysis.noCompletionTimeData") || "No completion time data available"}
          </p>
        )}
      </div>

      {/* Pattern Performance */}
      <div className='bg-white p-6 rounded-lg shadow'>
        <h3 className='text-lg font-semibold mb-4'>
          {t("analysis.taskAnalysis.performanceByPattern") || "Performance by Recurrence Pattern"}
        </h3>
        <div className='overflow-x-auto'>
          <table className='min-w-full divide-y divide-gray-200'>
            <thead className='bg-gray-50'>
              <tr>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.taskAnalysis.pattern") || "Pattern"}
                </th>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.taskAnalysis.totalTasksLabel") || "Total Tasks"}
                </th>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.taskAnalysis.completed") || "Completed"}
                </th>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.taskAnalysis.completionRateLabel") || "Completion Rate"}
                </th>
              </tr>
            </thead>
            <tbody className='bg-white divide-y divide-gray-200'>
              {patternStats.map((pattern) => (
                <tr key={pattern.pattern}>
                  <td className='px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900 capitalize'>
                    {pattern.pattern}
                  </td>
                  <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                    {pattern.total}
                  </td>
                  <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                    {pattern.completed}
                  </td>
                  <td className='px-6 py-4 whitespace-nowrap text-sm'>
                    <div className='flex items-center gap-2'>
                      <div className='flex-1 bg-gray-200 rounded-full h-2 max-w-[100px]'>
                        <div
                          className={`h-2 rounded-full ${
                            pattern.completionRate >= 80
                              ? "bg-green-500"
                              : pattern.completionRate >= 50
                              ? "bg-orange-500"
                              : "bg-red-500"
                          }`}
                          style={{
                            width: `${Math.min(pattern.completionRate, 100)}%`,
                          }}
                        />
                      </div>
                      <span
                        className={`font-medium ${
                          pattern.completionRate >= 80
                            ? "text-green-600"
                            : pattern.completionRate >= 50
                            ? "text-orange-600"
                            : "text-red-600"
                        }`}
                      >
                        {pattern.completionRate.toFixed(1)}%
                      </span>
                    </div>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>
    </>
  );
}

